#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>


#include "model/device_list.h"

//-------------------------------- private function definition -------------------------------------------------------
static error_code_t device_list_new_device(kms_device_t **dev_ptr, const char *identifier,
		const kms_credentials_t *credentials);

static void device_list_free_device(kms_device_t *device);

//--------------------------------------------------------------------------------------------------------------------

//-------------------------------- private attributes ----------------------------------------------------------------
static kms_device_t *first_in_list=NULL;

//--------------------------------------------------------------------------------------------------------------------

//-------------------------------- public functions ------------------------------------------------------------------
error_code_t device_list_add_no_credentials(const char *identifier)
{
	return device_list_add(identifier,NULL);
}

error_code_t device_list_add(const char *identifier, const kms_credentials_t *credentials)
{
	kms_device_t *device;
	error_code_t result;

	//--- create it -----
	result=device_list_new_device(&device, identifier,credentials);
	if (result==RESULT_OK)
	{
		//--- add it to the list ---

		//update forward pointer
		device->next_in_list_ptr=first_in_list;
		first_in_list=device;

		//update backward pointer
		if (device->next_in_list_ptr!=NULL)
			device->next_in_list_ptr->previous_in_list_ptr=device;
	}

	return result;
}

void device_list_remove_and_free(kms_device_t *device)
{
	kms_device_t *previous;
	kms_device_t *next;

	previous=device->previous_in_list_ptr;
	next=device->next_in_list_ptr;

	if (previous!=NULL)
		//we have a previous element -> link it forward to our next
		previous->next_in_list_ptr=next;
	else
		//we are the first in the list -> make our next one the first in the list
		first_in_list=next;

	//update the backwards link of the elemnt succeeding us of there is any
	if (next!=NULL)
		next->previous_in_list_ptr=previous;

	//remove us from memory
	device_list_free_device(device);
}

void device_list_clear(void)
{
	kms_device_t *tmp;

	while (first_in_list!=NULL)
	{
		tmp=first_in_list;
		first_in_list=tmp->next_in_list_ptr;
		device_list_free_device(tmp);
	}
}

kms_device_t *device_list_get_device(const char *sysfs_path, const char *devnode_path)
{
	kms_device_t *device;

	device=first_in_list;
	while (device!=NULL)
	{
		if (device->id_type==DEVNODE_PATH && devnode_path!=NULL)
		{
			if (fnmatch(device->identifier, devnode_path, FNM_PATHNAME|FNM_NOESCAPE)==0)
				return device;
		}
		else if (device->id_type==SYSFS_PATH && sysfs_path!=NULL)
		{
			if (fnmatch(device->identifier, sysfs_path, FNM_PATHNAME|FNM_NOESCAPE)==0)
				return device;
		}

		device=device->next_in_list_ptr;
	}

	return NULL;
}

kms_device_t *device_list_get_first_device(void)
{
	return first_in_list;
}

kms_device_t *device_list_get_next_device(kms_device_t *device)
{
	if (device==NULL)
		return NULL;

	return device->next_in_list_ptr;
}

bool device_list_is_empty(void)
{
	return first_in_list==NULL;
}

//--------------------------------------------------------------------------------------------------------------------

//------------------------------------ private functions -------------------------------------------------------------
static error_code_t device_list_new_device(kms_device_t **dev_ptr, const char *identifier,
		const kms_credentials_t *credentials)
{
	kms_device_t *device;
	size_t size;
	device_id_type_t id_type;

	//check identifier type
	if (strstr(identifier,"/sys")==identifier)
		id_type=SYSFS_PATH;
	else if (strstr(identifier,"/dev")==identifier)
		id_type=DEVNODE_PATH;
	else
		return RESULT_INVALID;

	//allocate memory
	size=sizeof(kms_device_t)+strlen(identifier)+1;
	device=malloc(size);
	if (device==NULL)
		return RESULT_NORESOURCES;

	strcpy(device->identifier, identifier);
	device->id_type=id_type;
	device->apply_credentials=(credentials!=NULL);
	if (credentials!=NULL)
		memcpy(&device->credentials,credentials,sizeof(kms_credentials_t));
	device->next_in_list_ptr=NULL;
	device->previous_in_list_ptr=NULL;

	*dev_ptr=device;
	return RESULT_OK;
}

static void device_list_free_device(kms_device_t *device)
{
	free(device);
}
//--------------------------------------------------------------------------------------------------------------------
